unit unicmain;
{
*****************************************************************
* Component naming conventions:                                 *
*  Bevel      : Bevel1, Bevel2,...                              *
*  Button     : BtnXxx             Ex.: BtnLenConvert           *
*  Edit       : EdXxx              Ex.: EdLength                *
*  Form       : FormXxx            Ex.: FormMain                *
*  Label      : LblXxx             Ex.: LblMM                   *
*  RadioGroup : RGXxxx             Ex.: RGLength                *
*  MenuItem   : MenuType1, MenuType2,... for first level        *
*                                  Ex.: MenuFile, MenuHelp      *
*               Type1Action1, TypeAction2,... for second level  *
*                                  Ex.: File2Decimals, FileExit *
*****************************************************************
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls, Menus;

type
  TFormMain = class(TForm)
    LblMM: TLabel;
    LblCM: TLabel;
    LblMeter: TLabel;
    LblKM: TLabel;
    LblInch: TLabel;
    LblFoot: TLabel;
    LblYard: TLabel;
    LblMile: TLabel;
    LblCC: TLabel;
    LblLiter: TLabel;
    LblM3: TLabel;
    LblInch3: TLabel;
    LblGalUS: TLabel;
    LblGalUK: TLabel;
    LblFoot3: TLabel;
    LblYard3: TLabel;
    EdLength: TEdit;
    EdVolume: TEdit;
    RGLength: TRadioGroup;
    RGDecimals: TRadioGroup;
    RGVolume: TRadioGroup;
    BtnLenConvert: TButton;
    BtnVolConvert: TButton;
    Bevel1: TBevel;
    Bevel2: TBevel;
    MainMenu1: TMainMenu;
    MenuFile: TMenuItem;
    File2Decimals: TMenuItem;
    File4Decimals: TMenuItem;
    FileSeparator1: TMenuItem;
    FileExit: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure BtnLenConvertClick(Sender: TObject);
    procedure File2DecimalsClick(Sender: TObject);
    procedure FileExitClick(Sender: TObject);
    procedure EdLengthKeyPress(Sender: TObject; var Key: Char);
    procedure RGDecimalsClick(Sender: TObject);
  private
    { Private declarations }
    procedure UnitsConvert(ConvType: integer);
    procedure ConvertShow(ConvType: integer);
    procedure SetLabelInvisible;
    function MakeString(MFactor: real; ConvType, LineNr: integer): string;
  public
    { Public declarations }
  end;

var
  FormMain: TFormMain;

implementation

{$R *.DFM}

const
  // Multiplication factors
  // 1 Unit_of_some_kind = X Base Units

  // Length: base unit = METER
  // In 1963, it was decreed that: The yard shall be 0.9144 m "exactly"
  // 1 foot = 12 inch
  // 1 yard = 3 ft
  // 1 mile = 1760 yards = 1609.344 m
  mfMM   = 0.001;    // 1 millimeter = 0.001 m
  mfCM   = 0.01;     // centimeter
  mfM    = 1;        // meter, the Base Unit
  mfKM   = 1000;     // kilometer
  mfInch = 0.0254;
  mfFoot = 0.3048;
  mfYard = 0.9144;
  mfMile = 1609.344;

  // Volume: base unit = CUBIC METER (m3)
  // Inch3, Foot3 and Yard3 are derived directly
  // from the lenght units definitions.
  // In the 1970's, it was decreed that:
  // 1 Gallon UK = 4.54609 Liter "exactly"
  // 1 Gallon US = 3.785411784 Liter "exactly"
  mfCC      = 0.000001;       // 1 cc = 0.000001 m3
  mfLiter   = 0.001;
  mfM3      = 1;              // cubic meter, the Base Unit
  mfInch3   = 0.000016387064;
  mfFoot3   = 0.028316846592;
  mfYard3   = 0.764554857984;
  mfGalUK   = 0.00454609;
  mfGalUS   = 0.003785411784;

var
  BU: Real; // global variable "Base Units"

procedure TFormMain.FormCreate(Sender: TObject);
begin
  RGLength.ItemIndex := 2;  // meter
  RGVolume.ItemIndex := 2;  // cubic meter
  // At start of program, do first "automatic" conversions
  UnitsConvert(0); // do the first length conversion
  UnitsConvert(1); // do the first volume conversion
end;

procedure TFormMain.BtnLenConvertClick(Sender: TObject);
begin
  if (Sender = BtnLenConvert) or (Sender = RGLength) then
    UnitsConvert(0)
  else
    UnitsConvert(1);
end;

procedure TFormMain.RGDecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := (RGDecimals.ItemIndex = 0);
  File4Decimals.Checked := (RGDecimals.ItemIndex = 1);
  UnitsConvert(0);
  UnitsConvert(1);
end;

{ Convert contents of Edit box to "base units".
  Convert to other units, show results in LABELs. }
procedure TFormMain.UnitsConvert(ConvType: integer);
var
  Amount,            // amount to convert
  MFactor: real;     // multiplication factor
begin
  try
    case ConvType of
      0: begin
          Amount := StrToFloat(EdLength.Text);
          case RGLength.ItemIndex of
            0 : MFactor := mfMM;
            1 : MFactor := mfCM;
            2 : MFactor := mfM;
            3 : MFactor := mfKM;
            4 : MFactor := mfInch;
            5 : MFactor := mfFoot;
            6 : MFactor := mfYard;
            7 : MFactor := mfMile;
          end;
         end;
      1: begin
          Amount := StrToFloat(EdVolume.Text);
          case RGVolume.ItemIndex of
            0 : MFactor := mfCC;
            1 : MFactor := mfLiter;
            2 : MFactor := mfM3;
            3 : MFactor := mfInch3;
            4 : MFactor := mfGalUS;
            5 : MFactor := mfGalUK;
            6 : MFactor := mfFoot3;
            7 : MFactor := mfYard3;
          end;
         end;
    end;
    BU := Amount * MFactor;
    // Convert to other length units and show results
    ConvertShow(ConvType);
    // Hide LABEL next to selected RADIOBUTTON
    SetLabelInvisible;
  except
    on EConvertError do
      MessageDlg('Illegal input!', mtWarning, [mbOk], 0);
  end;
end;

{ Multiply value BU (Base Units) with conversion factors,
  convert to STRING-format, show results in LABELS        }
procedure TFormMain.ConvertShow(ConvType: integer);
begin
  if (ConvType = 0) then begin
    LblMM.Caption    := MakeString(mfMM,   0, 0);
    LblCM.Caption    := MakeString(mfCM,   0, 1);
    LblMeter.Caption := MakeString(mfM,    0, 2);
    LblKM.Caption    := MakeString(mfKM,   0, 3);
    LblInch.Caption  := MakeString(mfInch, 0, 4);
    LblFoot.Caption  := MakeString(mfFoot, 0, 5);
    LblYard.Caption  := MakeString(mfYard, 0, 6);
    LblMile.Caption  := MakeString(mfMile, 0, 7);
  end
  else begin
    LblCC.Caption    := MakeString(mfCC,    1, 0);
    LblLiter.Caption := MakeString(mfLiter, 1, 1);
    LblM3.Caption    := MakeString(mfM3,    1, 2);
    LblInch3.Caption := MakeString(mfInch3, 1, 3);
    LblGalUS.Caption := MakeString(mfGalUS, 1, 4);
    LblGalUK.Caption := MakeString(mfGalUK, 1, 5);
    LblFoot3.Caption := MakeString(mfFoot3, 1, 6);
    LblYard3.Caption := MakeString(mfYard3, 1, 7);
  end;
end;

function TFormMain.MakeString(MFactor: real; ConvType, LineNr: integer): string;
var
  Other_Units, FormatStr, UnitName: string;
begin
  if RGDecimals.ItemIndex = 0 then
    FormatStr := '0.00'
  else
    FormatStr := '0.0000';

  Other_Units:= FormatFloat(FormatStr, BU / MFactor);

  // Make the unitname equal to the Radiogroup item
  case ConvType of
    0: UnitName := RGLength.Items[LineNr];
    1: UnitName := RGVolume.Items[LineNr];
  end;
  MakeString := Format('%15s%', [Other_Units]) + ' ' + UnitName;
end;

{ Hide LABEL next to selected RADIOBUTTON,
  make other LABELs visible               }
procedure TFormMain.SetLabelInvisible;
begin
  LblMM.Visible    := (RGLength.ItemIndex <> 0);
  LblCM.Visible    := (RGLength.ItemIndex <> 1);
  LblMeter.Visible := (RGLength.ItemIndex <> 2);
  LblKM.Visible    := (RGLength.ItemIndex <> 3);
  LblInch.Visible  := (RGLength.ItemIndex <> 4);
  LblFoot.Visible  := (RGLength.ItemIndex <> 5);
  LblYard.Visible  := (RGLength.ItemIndex <> 6);
  LblMile.Visible  := (RGLength.ItemIndex <> 7);

  LblCC.Visible    := (RGVolume.ItemIndex <> 0);
  LblLiter.Visible := (RGVolume.ItemIndex <> 1);
  LblM3.Visible    := (RGVolume.ItemIndex <> 2);
  LblInch3.Visible := (RGVolume.ItemIndex <> 3);
  LblGalUS.Visible := (RGVolume.ItemIndex <> 4);
  LblGalUK.Visible := (RGVolume.ItemIndex <> 5);
  LblFoot3.Visible := (RGVolume.ItemIndex <> 6);
  LblYard3.Visible := (RGVolume.ItemIndex <> 7);
end;

{ When clicked on a menu item for 2 or 4 decimals:
  update the check marks in the menu,
  select the correct RADIOBUTTON,
  execute both the conversion routines             }
procedure TFormMain.File2DecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := (Sender = File2Decimals);
  File4Decimals.Checked := (Sender = File4Decimals);
  if File2Decimals.Checked then
    RGDecimals.ItemIndex := 0
  else
    RGDecimals.ItemIndex := 1;
  UnitsConvert(0);
  UnitsConvert(1);
end;

{ When one of the Edits has the focus, and a key is pressed:
  if it was the ENTER key, the conversion has to be executed;
  if it was the COMMA key and Windows is set up for
  numerical format "decimal point", this comma is converted to a POINT;
  if it was the POINT key and Windows' decimal symbol is set
  to COMMA, convert this POINT to a COMMA.                   }
procedure TFormMain.EdLengthKeyPress(Sender: TObject; var Key: Char);
var
  ConvType: integer;
begin
  if Sender = EdLength then ConvType := 0
  else ConvType := 1;

  if Key = #13 then UnitsConvert(ConvType)  // ENTER was pressed
  else
    if (Key = ',') or (Key = '.') then Key := DecimalSeparator;
end;

procedure TFormMain.FileExitClick(Sender: TObject);
begin
  Close;
end;

end.
