unit unicmain;
{
*****************************************************************
* Component naming conventions:                                 *
*  Bevel      : Bevel1, Bevel2,...                              *
*  Button     : BtnXxx             Ex.: BtnLenConvert           *
*  Edit       : EdXxx              Ex.: EdLength                *
*  Form       : FormXxx            Ex.: FormMain                *
*  Label      : LblXxx             Ex.: LblMM                   *
*  RadioGroup : RGXxxx             Ex.: RGLength                *
*  MenuItem   : MenuType1, MenuType2,... for first level        *
*                                  Ex.: MenuFile, MenuHelp      *
*               Type1Action1, TypeAction2,... for second level  *
*                                  Ex.: File2Decimals, FileExit *
*****************************************************************
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls, Menus;

type
  TFormMain = class(TForm)
    LblMM: TLabel;
    LblCM: TLabel;
    LblMeter: TLabel;
    LblKM: TLabel;
    LblInch: TLabel;
    LblFoot: TLabel;
    LblYard: TLabel;
    LblMile: TLabel;
    LblCC: TLabel;
    LblLiter: TLabel;
    LblM3: TLabel;
    LblInch3: TLabel;
    LblGalUS: TLabel;
    LblGalUK: TLabel;
    LblFoot3: TLabel;
    LblYard3: TLabel;
    EdLength: TEdit;
    EdVolume: TEdit;
    RGLength: TRadioGroup;
    RGDecimals: TRadioGroup;
    RGVolume: TRadioGroup;
    BtnLenConvert: TButton;
    BtnVolConvert: TButton;
    Bevel1: TBevel;
    Bevel2: TBevel;
    MainMenu1: TMainMenu;
    MenuFile: TMenuItem;
    File2Decimals: TMenuItem;
    File4Decimals: TMenuItem;
    FileSeparator1: TMenuItem;
    FileExit: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure BtnLenConvertClick(Sender: TObject);
    procedure RGLengthClick(Sender: TObject);
    procedure RGDecimalsClick(Sender: TObject);
    procedure File2DecimalsClick(Sender: TObject);
    procedure File4DecimalsClick(Sender: TObject);
    procedure RGVolumeClick(Sender: TObject);
    procedure BtnVolConvertClick(Sender: TObject);
    procedure EdLengthKeyPress(Sender: TObject; var Key: Char);
    procedure EdVolumeKeyPress(Sender: TObject; var Key: Char);
    procedure FileExitClick(Sender: TObject);
  private
    { Private declarations }
    procedure LengthConvert;
    procedure VolumeConvert;
    procedure ConvertLengthShow;
    procedure ConvertVolumeShow;
    procedure SetLabelInvisible;
  public
    { Public declarations }
  end;

var
  FormMain: TFormMain;

implementation

{$R *.DFM}

const
  // Multiplication factors
  // 1 Unit_of_some_kind = X Base Units

  // Length: base unit = METER
  mfMM   = 0.001;    // 1 millimeter = 0.001 m
  mfCM   = 0.01;     // centimeter
  mfM    = 1;        // meter, the Base Unit
  mfKM   = 1000;     // kilometer
  // 1 foot = 12 inch
  // 1 yard = 3 ft
  // 1 mile = 1760 yards = 1609.344 m
  // In 1963, it was decreed that: The yard shall be 0.9144 m "exactly"
  mfInch = 0.0254;
  mfFoot = 0.3048;
  mfYard = 0.9144;
  mfMile = 1609.344;

  // Volume: base unit = CUBIC METER (m3)
  mfCC      = 0.000001;  // 1 cc (ml) = 0.000001 m3
  mfLiter   = 0.001;
  mfM3      = 1;         // cubic meter, the Base Unit
  // Derived from the lenght units definitions:
  mfInch3   = 0.000016387064;
  mfFoot3   = 0.028316846592;
  mfYard3   = 0.764554857984;
  // In the 1970's, it was decreed that:
  // 1 Gallon UK = 4.54609 Liter "exactly"
  // 1 Gallon US = 3.785411784 Liter "exactly"
  mfGalUK   = 0.00454609;
  mfGalUS   = 0.003785411784;

var
  BU: Real; // global variable "Base Units"

procedure TFormMain.FormCreate(Sender: TObject);
begin
  RGLength.ItemIndex := 2;  // meter
  RGVolume.ItemIndex := 2;  // cubic meter
  // At start of program, do first "automatic" conversion
  LengthConvert;
  VolumeConvert;
end;

procedure TFormMain.BtnLenConvertClick(Sender: TObject);
begin
  LengthConvert;
end;

procedure TFormMain.RGLengthClick(Sender: TObject);
begin
  LengthConvert;
end;

procedure TFormMain.RGDecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := (RGDecimals.ItemIndex = 0);
  File4Decimals.Checked := (RGDecimals.ItemIndex = 1);
  LengthConvert;
  VolumeConvert;
end;

procedure TFormMain.BtnVolConvertClick(Sender: TObject);
begin
  VolumeConvert;
end;

procedure TFormMain.RGVolumeClick(Sender: TObject);
begin
  VolumeConvert;
end;

{ Convert contents of EdLength to "base units",
  convert to other units,
  show results in LABELs.                      }
procedure TFormMain.LengthConvert;
var
  Amount,            // amount to convert
  MFactor: real;     // multiplication factor
begin
  try
    // Convert text of EDIT to numerical value
    Amount := StrToFloat(EdLength.Text);
    { Conversion factor depends from selected
      button in RADIOGROUP RGLength           }
    case RGLength.ItemIndex of
      0 : MFactor := mfMM;
      1 : MFactor := mfCM;
      2 : MFactor := mfM;
      3 : MFactor := mfKM;
      4 : MFactor := mfInch;
      5 : MFactor := mfFoot;
      6 : MFactor := mfYard;
      7 : MFactor := mfMile;
    end;
    BU := Amount * mfactor;
    // Convert to other length units and show results
    ConvertLengthShow;
    // Hide LABEL next to selected RADIOBUTTON
    SetLabelInvisible;
  except
    { When there is a conversion error,
      we end up here, IMMEDIATELY after trying
      to execute the function StrToFloat()     }
    on EConvertError do
      MessageDlg('Illegal length input!', mtWarning, [mbOk], 0);
  end;
end;

{ Convert contents of EdVolume to "base units",
  convert to other units,
  show results in LABELs.                      }
procedure TFormMain.VolumeConvert;
var
  Amount,            // amount to convert
  MFactor: real;     // multiplication factor
begin
  try
    // Convert text of EDIT to numerical value
    Amount := StrToFloat(EdVolume.Text);
    { Conversion factor depends from selected
      button in RADIOGROUP RGVolume           }
    case RGVolume.ItemIndex of
      0 : MFactor := mfCC;
      1 : MFactor := mfLiter;
      2 : MFactor := mfM3;
      3 : MFactor := mfInch3;
      4 : MFactor := mfGalUS;
      5 : MFactor := mfGalUK;
      6 : MFactor := mfFoot3;
      7 : MFactor := mfYard3;
    end;
    BU := Amount * MFactor;
    // Convert to other length units and show results
    ConvertVolumeShow;
    // Hide LABEL next to selected RADIOBUTTON
    SetLabelInvisible;
  except
    { When there is a conversion error,
      we end up here, IMMEDIATELY after trying
      to execute the function StrToFloat()     }
    on EConvertError do
      MessageDlg('Illegal volume input!', mtWarning, [mbOk], 0);
  end;
end;

{ Multiply value in BU (Base Units) with conversion factors,
  convert to STRING-format and
  show results in corresponding LABELS                       }
procedure TFormMain.ConvertLengthShow;
var
  FStr: string; // format-string
begin
  { Number of decimals depends on selected button
    in RADIOGROUP RGDecimals                      }
  if RGDecimals.ItemIndex = 0 then
    FStr := '0.00'
  else
    FStr := '0.0000';

  { Convert from Base Units to other units,
    format, append the unit names,
    add spaces as to obtain equal lengths,
    put resulting strings in CAPTIONs of LABELs }
  LblMM.Caption    := FormatFloat(FStr, BU / mfMM)   + ' mm  ';
  LblCM.Caption    := FormatFloat(FStr, BU / mfCM)   + ' cm  ';
  LblMeter.Caption := FormatFloat(FStr, BU / mfM)    + ' m   ';
  LblKM.Caption    := FormatFloat(FStr, BU / mfKM)   + ' km  ';
  LblInch.Caption  := FormatFloat(FStr, BU / mfInch) + ' inch';
  LblFoot.Caption  := FormatFloat(FStr, BU / mfFoot) + ' foot';
  LblYard.Caption  := FormatFloat(FStr, BU / mfYard) + ' yard';
  LblMile.Caption  := FormatFloat(FStr, BU / mfMile) + ' mile';
end;

{ Multiply value in BU (Base Units) with conversion factors,
  convert to STRING-format and
  show results in corresponding LABELS                      }
procedure TFormMain.ConvertVolumeShow;
var
  FStr: string; // format-string
begin
  { Number of decimals depends on selected button
    in RADIOGROUP RGDecimals                     }
  if RGDecimals.ItemIndex = 0 then
    FStr := '0.00'
  else
    FStr := '0.0000';

  { Convert from Base Units to other units,
    format, append the unit names,
    add spaces as to obtain equal lengths,
    put resulting strings in CAPTIONs of LABELs }
  LblCC.Caption    := FormatFloat(FStr, BU / mfCC)    + ' cc   ';
  LblLiter.Caption := FormatFloat(FStr, BU / mfLiter) + ' liter';
  LblM3.Caption    := FormatFloat(FStr, BU / mfM3)    + ' m3   ';
  LblInch3.Caption := FormatFloat(FStr, BU / mfInch3) + ' inch3';
  LblGalUS.Caption := FormatFloat(FStr, BU / mfGalUS) + ' galUS';
  LblGalUK.Caption := FormatFloat(FStr, BU / mfGalUK) + ' galUK';
  LblFoot3.Caption := FormatFloat(FStr, BU / mfFoot3) + ' foot3';
  LblYard3.Caption := FormatFloat(FStr, BU / mfYard3) + ' yard3';
end;

{ Hide LABEL next to selected RADIOBUTTON,
  make other LABELs visible               }
procedure TFormMain.SetLabelInvisible;
begin
  LblMM.Visible    := (RGLength.ItemIndex <> 0);
  LblCM.Visible    := (RGLength.ItemIndex <> 1);
  LblMeter.Visible := (RGLength.ItemIndex <> 2);
  LblKM.Visible    := (RGLength.ItemIndex <> 3);
  LblInch.Visible  := (RGLength.ItemIndex <> 4);
  LblFoot.Visible  := (RGLength.ItemIndex <> 5);
  LblYard.Visible  := (RGLength.ItemIndex <> 6);
  LblMile.Visible  := (RGLength.ItemIndex <> 7);

  LblCC.Visible    := (RGVolume.ItemIndex <> 0);
  LblLiter.Visible := (RGVolume.ItemIndex <> 1);
  LblM3.Visible    := (RGVolume.ItemIndex <> 2);
  LblInch3.Visible := (RGVolume.ItemIndex <> 3);
  LblGalUS.Visible := (RGVolume.ItemIndex <> 4);
  LblGalUK.Visible := (RGVolume.ItemIndex <> 5);
  LblFoot3.Visible := (RGVolume.ItemIndex <> 6);
  LblYard3.Visible := (RGVolume.ItemIndex <> 7);
end;

{ Update the check marks in the menu,
  select the first RADIOBUTTON of RGDecimals
  and execute the conversion routine             }
procedure TFormMain.File2DecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := TRUE;
  File4Decimals.Checked := FALSE;
  RGDecimals.ItemIndex := 0;
  LengthConvert;
  VolumeConvert;
end;

{ Update the check marks in the menu,
  select the second RADIOBUTTON of RGDecimals
  and execute the conversion routine             }
procedure TFormMain.File4DecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := FALSE;
  File4Decimals.Checked := TRUE;
  RGDecimals.ItemIndex := 1;
  LengthConvert;
  VolumeConvert;
end;

{ When EdLength has the focus, and a key is pressed:
  if it was the ENTER key, the conversion has to be executed;
  if it was the COMMA key and Windows is set up for
  numerical format "decimal point", this comma is converted
  to a POINT;
  if it was the POINT key and Windows' decimal symbol is set
  to COMMA, convert this POINT to a COMMA.                   }
procedure TFormMain.EdLengthKeyPress(Sender: TObject; var Key: Char);
begin
  // the code for the ENTER key is 13
  if Key = #13 then
    LengthConvert
  { DecimalSeparator is a GLOBAL variabele in Delphi
    depending from Windows' "Regional Settings"      }
  else
    if (Key = ',') and (DecimalSeparator = '.') then
      Key := '.'
    else
      if (Key = '.') and (DecimalSeparator = ',') then
        Key := ',';
end;

{ When EdVolume has the focus, and a key is pressed:
  if it was the ENTER key, the conversion has to be executed;
  if it was the COMMA key and Windows is set up for
  numerical format "decimal point", this comma is converted
  to a POINT;
  if it was the POINT key and Windows' decimal symbol is set
  to COMMA, convert this POINT to a COMMA.                   }
procedure TFormMain.EdVolumeKeyPress(Sender: TObject; var Key: Char);
begin
  // the code for the ENTER key is 13
  if Key = #13 then
    VolumeConvert
  { DecimalSeparator is a GLOBAL variabele in Delphi
    depending from Windows' "Regional Settings"      }
  else
    if (Key = ',') and (DecimalSeparator = '.') then
      Key := '.'
    else
      if (Key = '.') and (DecimalSeparator = ',') then
        Key := ',';
end;

// Close the application
procedure TFormMain.FileExitClick(Sender: TObject);
begin
  { Closing the MAIN form of an application will
    automatically close the application         }
  Close;
end;

end.
