unit euroform;
{
*****************************************************************
* Component naming conventions:                                 *
*  Bevel      : Bevel1, Bevel2,...                              *
*  Button     : BtnXxx             Ex.: BtnConvert              *
*  Edit       : EdXxx              Ex.: EdInput                 *
*  Form       : FormXxx            Ex.: FormMain                *
*  Label      : LblXxx             Ex.: LblEuro                 *
*  RadioGroup : RGXxxx             Ex.: RGCurrency              *
*  MenuItem   : MenuType1, MenuType2,... for first level        *
*                                  Ex.: FileMenu, HelpMenu      *
*               Type1Action1, TypeAction2,... for second level  *
*                                  Ex.: File2Decimals, FileExit *
*****************************************************************
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls, Menus;

type
  TFormMain = class(TForm)
    LblEuro: TLabel;
    LblATS: TLabel;
    LblBEF: TLabel;
    LblDEM: TLabel;
    LblESP: TLabel;
    LblFIM: TLabel;
    LblFRF: TLabel;
    LblIEP: TLabel;
    LblITL: TLabel;
    LblNLG: TLabel;
    LblPTE: TLabel;
    EdInput: TEdit;
    BtnConvert: TButton;
    RGCurrency: TRadioGroup;
    Bevel1: TBevel;
    RGDecimals: TRadioGroup;
    MainMenu1: TMainMenu;
    MenuFile: TMenuItem;
    FileExit: TMenuItem;
    File2Decimals: TMenuItem;
    File4Decimals: TMenuItem;
    FileSeparator: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure BtnConvertClick(Sender: TObject);
    procedure RGCurrencyClick(Sender: TObject);
    procedure RGDecimalsClick(Sender: TObject);
    procedure File2DecimalsClick(Sender: TObject);
    procedure File4DecimalsClick(Sender: TObject);
    procedure FileExitClick(Sender: TObject);
    procedure EdInputKeyPress(Sender: TObject; var Key: Char);
  private
    { Private declarations }
    procedure CurrConvert;
    procedure ConvertShow;
    procedure SetLabelInvisible;
  public
    { Public declarations }
  end;

var
  FormMain: TFormMain;

implementation

{$R *.DFM}

const
  // Conversion factors for currencies
  // 1 EURO = X currency units
  cfATS  = 13.7603;
  cfBEF  = 40.3399;
  cfDEM  = 1.95583;
  cfESP  = 166.386;
  cfFIM  = 5.94573;
  cfFRF  = 6.55957;
  cfIEP  = 0.787564;
  cfITL  = 1936.27;
  cfNLG  = 2.20371;
  cfPTE  = 200.482;

var
  Euro: Real; // global variable for amount in Euro

procedure TFormMain.BtnConvertClick(Sender: TObject);
begin
  CurrConvert;
end;

procedure TFormMain.RGCurrencyClick(Sender: TObject);
begin
  CurrConvert;
end;

procedure TFormMain.RGDecimalsClick(Sender: TObject);
begin
  CurrConvert;
end;

// At start of program, do first "automatic" conversion
procedure TFormMain.FormCreate(Sender: TObject);
begin
  CurrConvert;
end;

{ Convert contents of the EDIT to EURO,
  convert to other currencies,
  show results in LABELs.               }
procedure TFormMain.CurrConvert;
var
  Amount,            // amount to convert
  CFactor: real;     // conversion factor
begin
  try
    // Convert text of EDIT to numerical value
    Amount := StrToFloat(EdInput.Text);
    { Conversion factor depends from selected
      button in RADIOGROUP RGCurrency         }
    case RGCurrency.ItemIndex of
      1 : CFactor := cfATS;
      2 : CFactor := cfBEF;
      3 : CFactor := cfDEM;
      4 : CFactor := cfESP;
      5 : CFactor := cfFIM;
      6 : CFactor := cfFRF;
      7 : CFactor := cfIEP;
      8 : CFactor := cfITL;
      9 : CFactor := cfNLG;
      10: CFactor := cfPTE;
    else
      CFactor := 1; // first RadioButton is Euro
    end;
    Euro := Amount / CFactor;   // plain arithmetics...
    // Convert to other currencies and show results
    ConvertShow;
    // Hide LABEL next to selected RADIOBUTTON
    SetLabelInvisible;
  except
    { When there is a conversion error,
      we end up here, IMMEDIATELY after trying
      to execute the function StrToFloat()     }
    on EConvertError do
      MessageDlg('Illegal input!', mtWarning, [mbOk], 0);
  end;
end;

{ Multiply value in Euro with conversion factors,
  convert to STRING-format and
  show results in correspondingLABELS             }
procedure TFormMain.ConvertShow;
var
  FStr: string; // format-string
begin
  { Number of decimals depends on selected button
    in RADIOGROUP RGDecimals                      }
  if RGDecimals.ItemIndex = 0 then
    FStr := '0.00'
  else
    FStr := '0.0000';

  { Convert from Euro to other currencies,
    format, append the currency names,
    add spaces as to obtain equal lengths,
    put resulting strings in CAPTIONs of LABELs }
  LblEuro.Caption := FormatFloat(FStr, Euro)        + ' EURO   ';
  LblATS.Caption := FormatFloat(FStr, Euro * cfATS) + ' ATS    ';
  LblBEF.Caption := FormatFloat(FStr, Euro * cfBEF) + ' BEF/LUF';
  LblDEM.Caption := FormatFloat(FStr, Euro * cfDEM) + ' DEM    ';
  LblESP.Caption := FormatFloat(FStr, Euro * cfESP) + ' ESP    ';
  LblFIM.Caption := FormatFloat(FStr, Euro * cfFIM) + ' FIM    ';
  LblFRF.Caption := FormatFloat(FStr, Euro * cfFRF) + ' FRF    ';
  LblIEP.Caption := FormatFloat(FStr, Euro * cfIEP) + ' IEP    ';
  LblITL.Caption := FormatFloat(FStr, Euro * cfITL) + ' ITL    ';
  LblNLG.Caption := FormatFloat(FStr, Euro * cfNLG) + ' NLG    ';
  LblPTE.Caption := FormatFloat(FStr, Euro * cfPTE) + ' PTE    ';
end;

{ Hide LABEL next to selected RADIOBUTTON,
  make other LABELs visible               }
procedure TFormMain.SetLabelInvisible;
begin
  LblEuro.Visible := (RGCurrency.ItemIndex <> 0);
  LblATS.Visible  := (RGCurrency.ItemIndex <> 1);
  LblBEF.Visible  := (RGCurrency.ItemIndex <> 2);
  LblDEM.Visible  := (RGCurrency.ItemIndex <> 3);
  LblESP.Visible  := (RGCurrency.ItemIndex <> 4);
  LblFIM.Visible  := (RGCurrency.ItemIndex <> 5);
  LblFRF.Visible  := (RGCurrency.ItemIndex <> 6);
  LblIEP.Visible  := (RGCurrency.ItemIndex <> 7);
  LblITL.Visible  := (RGCurrency.ItemIndex <> 8);
  LblNLG.Visible  := (RGCurrency.ItemIndex <> 9);
  LblPTE.Visible  := (RGCurrency.ItemIndex <> 10);
end;

{ Update the check marks in the menu,
  select the first RADIOBUTTON of DecimalsRadioG
  and execute the conversion routine             }
procedure TFormMain.File2DecimalsClick(Sender: TObject);
begin
  File2Decimals.Checked := TRUE;
  File4Decimals.Checked := FALSE;
  RGDecimals.ItemIndex := 0;
  CurrConvert;
end;

{ Update the check marks in the menu,
  select the second RADIOBUTTON of DecimalsRadioG
  and execute the conversion routine             }
procedure TFormMain.File4DecimalsClick(Sender: TObject);
begin
  File4Decimals.Checked := TRUE;
  File2Decimals.Checked := FALSE;
  RGDecimals.ItemIndex := 1;
  CurrConvert;
end;

// Close the application
procedure TFormMain.FileExitClick(Sender: TObject);
begin
  { Closing the MAIN form of an application will
    automatically close the application         }
  Close;
end;

{ When EdInput has the focus, and a key is pressed:
  if it was the ENTER key, the conversion has to be executed;
  if it was the COMMA key and Windows is set up for
  numerical format "decimal point", this comma is converted
  to a POINT;
  if it was the POINT key and Windows' decimal symbol is set
  to COMMA, convert this POINT to a COMMA.                   }
procedure TFormMain.EdInputKeyPress(Sender: TObject; var Key: Char);
begin
  // the code for the ENTER key is 13
  if Key = #13 then
    CurrConvert
  { DecimalSeparator is a GLOBAL variabele in Delphi
    depending from Windows' "Regional Settings"      }
  else
    if (Key = ',') and (DecimalSeparator = '.') then
      Key := '.'
    else
      if (Key = '.') and (DecimalSeparator = ',') then
        Key := ',';
end;

end.
